// Initialize GSAP ScrollTrigger
gsap.registerPlugin(ScrollTrigger);

document.addEventListener('DOMContentLoaded', () => {

    // --- 1. Global Section Reveals (Smooth Fade In) ---
    // Select all main sections that should fade in
    const sections = gsap.utils.toArray('section:not(.hero-section):not(.page-banner)');

    sections.forEach(section => {
        gsap.fromTo(section,
            { opacity: 0, y: 30 },
            {
                opacity: 1,
                y: 0,
                duration: 1,
                ease: 'power2.out',
                scrollTrigger: {
                    trigger: section,
                    start: 'top 85%', // Trigger when top of section hits 85% of viewport
                    toggleActions: 'play none none reverse' // Reverses on scroll up for "live" feel
                }
            }
        );
    });

    // --- 2. Parallax Effects for Page Banners ---
    // Only if .page-banner exists
    if (document.querySelector('.page-banner')) {
        gsap.to('.page-banner', {
            backgroundPosition: '50% 100%',
            ease: "none",
            scrollTrigger: {
                trigger: '.page-banner',
                start: 'top top',
                end: 'bottom top',
                scrub: true
            }
        });
    }

    // --- 3. Elegant Text Stagger ---
    // Targets: Display headings, section masks, and luxury card titles
    const textElements = gsap.utils.toArray('.display-1, .display-4, .display-5, .section-title-v2, .page-banner-title');

    textElements.forEach(text => {
        gsap.from(text, {
            scrollTrigger: {
                trigger: text,
                start: 'top 90%',
            },
            y: 40,
            opacity: 0,
            duration: 1.2,
            ease: 'power3.out'
        });
    });

    // --- 4. Image Reveal Masks ---
    // This looks for elements with .reveal-holder wrapper and .reveal-mask overlay
    const revealHolders = gsap.utils.toArray('.reveal-holder');

    revealHolders.forEach(holder => {
        const mask = holder.querySelector('.reveal-mask');
        const img = holder.querySelector('img');

        if (mask && img) {
            const tl = gsap.timeline({
                scrollTrigger: {
                    trigger: holder,
                    start: 'top 80%',
                }
            });

            // 1. Mask slides away
            tl.to(mask, {
                width: '0%',
                duration: 1.2,
                ease: 'power2.inOut'
            })
                // 2. Image scales down slightly for "arrival" effect
                .from(img, {
                    scale: 1.2,
                    duration: 1.4,
                    ease: 'power2.out'
                }, '-=1.2'); // Run simultaneously
        }
    });

    // --- 5. Luxury Card Staggers ---
    const grids = gsap.utils.toArray('.luxury-grid, .row');

    grids.forEach(grid => {
        // Find cards inside this specific grid
        const cards = grid.querySelectorAll('.luxury-card, .col-md-4 .card');

        if (cards.length > 0) {
            gsap.from(cards, {
                scrollTrigger: {
                    trigger: grid,
                    start: 'top 95%', // Earlier trigger for safety
                },
                y: 50,
                duration: 0.8,
                stagger: 0.2, // Time between each card animation
                ease: 'back.out(1.7)' // Subtle bounce
            });
        }
    });

    // --- 6. Navbar Link Hover Line Animation (Optional JS enhancement) ---
    // The CSS does most of this, but we can add a subtle scale
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('mouseenter', () => {
            // Optional GSAP specific micro-interactions
        });
    });

});