/**
 * Particle System (Canvas-based)
 * Creates floating particles in hero section
 * Lanka Matha Tours - About Page Advanced
 */

class ParticleSystem {
    constructor(canvasId, options = {}) {
        this.canvas = document.getElementById(canvasId);
        if (!this.canvas) return;

        this.ctx = this.canvas.getContext('2d');
        this.particles = [];
        this.animationId = null;

        // Configuration
        this.config = {
            particleCount: options.particleCount || 50,
            particleColor: options.particleColor || 'rgba(255, 255, 255, 0.5)',
            particleSize: options.particleSize || 3,
            speed: options.speed || 0.5,
            ...options
        };

        // Adjust particle count for mobile
        if (window.innerWidth < 768) {
            this.config.particleCount = Math.floor(this.config.particleCount / 2);
        }

        this.init();
    }

    init() {
        this.resize();
        this.createParticles();
        this.animate();
        this.handleResize();
    }

    resize() {
        this.canvas.width = this.canvas.offsetWidth;
        this.canvas.height = this.canvas.offsetHeight;
    }

    handleResize() {
        window.addEventListener('resize', () => {
            this.resize();
            this.createParticles(); // Recreate particles on resize
        });
    }

    createParticles() {
        this.particles = [];

        for (let i = 0; i < this.config.particleCount; i++) {
            this.particles.push({
                x: Math.random() * this.canvas.width,
                y: Math.random() * this.canvas.height,
                vx: (Math.random() - 0.5) * this.config.speed,
                vy: (Math.random() - 0.5) * this.config.speed,
                size: Math.random() * this.config.particleSize + 1,
                opacity: Math.random() * 0.5 + 0.2,
                rotation: Math.random() * 360,
                rotationSpeed: (Math.random() - 0.5) * 2
            });
        }
    }

    updateParticles() {
        this.particles.forEach(particle => {
            // Update position
            particle.x += particle.vx;
            particle.y += particle.vy;

            // Update rotation
            particle.rotation += particle.rotationSpeed;

            // Wrap around edges
            if (particle.x < 0) particle.x = this.canvas.width;
            if (particle.x > this.canvas.width) particle.x = 0;
            if (particle.y < 0) particle.y = this.canvas.height;
            if (particle.y > this.canvas.height) particle.y = 0;

            // Subtle opacity pulsing
            particle.opacity += Math.sin(Date.now() * 0.001) * 0.002;
            particle.opacity = Math.max(0.1, Math.min(0.7, particle.opacity));
        });
    }

    drawParticles() {
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);

        this.particles.forEach(particle => {
            this.ctx.save();

            // Set opacity
            this.ctx.globalAlpha = particle.opacity;

            // Move to particle position and rotate
            this.ctx.translate(particle.x, particle.y);
            this.ctx.rotate(particle.rotation * Math.PI / 180);

            // Draw particle (circle)
            this.ctx.fillStyle = this.config.particleColor;
            this.ctx.beginPath();
            this.ctx.arc(0, 0, particle.size, 0, Math.PI * 2);
            this.ctx.fill();

            this.ctx.restore();
        });
    }

    animate() {
        this.updateParticles();
        this.drawParticles();
        this.animationId = requestAnimationFrame(() => this.animate());
    }

    stop() {
        if (this.animationId) {
            cancelAnimationFrame(this.animationId);
            this.animationId = null;
        }
    }

    start() {
        if (!this.animationId) {
            this.animate();
        }
    }
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
    const prefersReducedMotion = window.matchMedia('(prefers-reduced-motion: reduce)').matches;

    if (!prefersReducedMotion) {
        const heroParticles = new ParticleSystem('hero-particles', {
            particleCount: 50,
            particleColor: 'rgba(255, 255, 255, 0.4)',
            particleSize: 3,
            speed: 0.3
        });

        window.heroParticles = heroParticles;
    }
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ParticleSystem;
}
