/**
 * Image Switcher
 * Changes images based on scroll position for Why Choose Us section
 * Lanka Matha Tours - About Page Advanced
 */

class ImageSwitcher {
    constructor() {
        this.container = document.querySelector('.choose-us-split');
        this.images = [];
        this.featureBlocks = [];
        this.currentIndex = 0;
        this.observer = null;
        this.progressLine = null;

        this.init();
    }

    init() {
        if (!this.container) return;

        this.findElements();
        this.preloadImages();
        this.setupObserver();
        this.setupProgressTracking();
    }

    findElements() {
        this.images = Array.from(document.querySelectorAll('.stack-img'));
        this.featureBlocks = Array.from(document.querySelectorAll('[data-feature-id]'));
        this.progressLine = document.querySelector('[data-scroll-progress]');
    }

    preloadImages() {
        this.images.forEach(img => {
            if (!img.complete) {
                img.loading = 'eager';
            }
        });
    }

    setupObserver() {
        if (!('IntersectionObserver' in window)) return;

        const options = {
            threshold: 0.5,
            rootMargin: '-20% 0px -20% 0px'
        };

        this.observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const featureId = entry.target.getAttribute('data-feature-id');
                    this.switchImage(featureId);
                    this.setActiveFeature(entry.target);
                }
            });
        }, options);

        this.featureBlocks.forEach(block => {
            this.observer.observe(block);
        });
    }

    switchImage(featureId) {
        // Remove active class from all images
        this.images.forEach(img => {
            img.classList.remove('active');
        });

        // Add active class to corresponding image
        const targetImage = document.querySelector(`[data-feature="${featureId}"]`);
        if (targetImage) {
            targetImage.classList.add('active');
        }
    }

    setActiveFeature(activeBlock) {
        // Remove active class from all feature blocks
        this.featureBlocks.forEach(block => {
            block.classList.remove('active');
        });

        // Add active class to current block
        activeBlock.classList.add('active');

        // Update timeline dot if exists
        const index = this.featureBlocks.indexOf(activeBlock);
        this.updateTimelineDots(index);
    }

    updateTimelineDots(activeIndex) {
        const dots = document.querySelectorAll('.timeline-dot');
        dots.forEach((dot, index) => {
            if (index === activeIndex) {
                dot.classList.add('active');
            } else {
                dot.classList.remove('active');
            }
        });
    }

    setupProgressTracking() {
        if (!this.progressLine) return;

        window.addEventListener('scroll', () => {
            this.updateProgress();
        });
    }

    updateProgress() {
        if (!this.featureBlocks.length || !this.progressLine) return;

        const firstBlock = this.featureBlocks[0];
        const lastBlock = this.featureBlocks[this.featureBlocks.length - 1];

        const firstTop = firstBlock.getBoundingClientRect().top + window.pageYOffset;
        const lastTop = lastBlock.getBoundingClientRect().top + window.pageYOffset;
        const scrolled = window.pageYOffset;

        const totalHeight = lastTop - firstTop;
        const progress = Math.max(0, Math.min(100, ((scrolled - firstTop) / totalHeight) * 100));

        this.progressLine.style.height = `${progress}%`;
    }

    // Manually switch to specific feature
    goToFeature(featureId) {
        this.switchImage(featureId);
        const featureBlock = document.querySelector(`[data-feature-id="${featureId}"]`);
        if (featureBlock) {
            this.setActiveFeature(featureBlock);
        }
    }

    // Get current active feature
    getCurrentFeature() {
        return this.featureBlocks.find(block => block.classList.contains('active'));
    }
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
    const imageSwitcher = new ImageSwitcher();
    window.imageSwitcher = imageSwitcher;
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ImageSwitcher;
}
