/**
 * Counter Animations
 * Animate numbers from 0 to target value with easing
 * Lanka Matha Tours - About Page Advanced
 */

class CounterAnimation {
    constructor(element, options = {}) {
        this.element = element;
        this.target = parseInt(element.getAttribute('data-count-to')) || 0;
        this.duration = options.duration || 2000;
        this.easing = options.easing || this.easeOutExpo;
        this.useCommas = options.useCommas !== undefined ? options.useCommas : true;
        this.suffix = element.getAttribute('data-suffix') || '';
        this.prefix = element.getAttribute('data-prefix') || '';

        this.hasRun = false;
    }

    easeOutExpo(t) {
        return t === 1 ? 1 : 1 - Math.pow(2, -10 * t);
    }

    easeOutCubic(t) {
        return 1 - Math.pow(1 - t, 3);
    }

    formatNumber(num) {
        if (this.useCommas) {
            return num.toLocaleString();
        }
        return num.toString();
    }

    animate() {
        if (this.hasRun) return;
        this.hasRun = true;

        const startTime = performance.now();
        const startValue = 0;

        const updateCounter = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / this.duration, 1);
            const easedProgress = this.easing(progress);

            const currentValue = Math.floor(startValue + (this.target - startValue) * easedProgress);

            this.element.textContent = this.prefix + this.formatNumber(currentValue) + this.suffix;

            if (progress < 1) {
                requestAnimationFrame(updateCounter);
            } else {
                // Ensure we end at exact target
                this.element.textContent = this.prefix + this.formatNumber(this.target) + this.suffix;
            }
        };

        requestAnimationFrame(updateCounter);
    }

    reset() {
        this.hasRun = false;
        this.element.textContent = this.prefix + '0' + this.suffix;
    }
}

class CounterController {
    constructor() {
        this.counters = [];
        this.observer = null;
        this.init();
    }

    init() {
        this.findCounters();
        this.setupObserver();
    }

    findCounters() {
        const counterElements = document.querySelectorAll('[data-count-to]');

        counterElements.forEach(element => {
            const counter = new CounterAnimation(element, {
                duration: parseInt(element.getAttribute('data-duration')) || 2000,
                useCommas: true
            });
            this.counters.push(counter);
        });
    }

    setupObserver() {
        if ('IntersectionObserver' in window) {
            const options = {
                threshold: 0.5,
                rootMargin: '0px'
            };

            this.observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const counter = this.counters.find(c => c.element === entry.target);
                        if (counter && !counter.hasRun) {
                            counter.animate();
                        }
                    }
                });
            }, options);

            this.counters.forEach(counter => {
                this.observer.observe(counter.element);
            });
        } else {
            // Fallback: animate all counters immediately
            this.counters.forEach(counter => counter.animate());
        }
    }

    // Manually trigger a counter
    triggerCounter(element) {
        const counter = this.counters.find(c => c.element === element);
        if (counter) {
            counter.animate();
        }
    }

    // Reset a counter
    resetCounter(element) {
        const counter = this.counters.find(c => c.element === element);
        if (counter) {
            counter.reset();
            if (this.observer) {
                this.observer.observe(element);
            }
        }
    }

    // Reset all counters
    resetAll() {
        this.counters.forEach(counter => {
            counter.reset();
            if (this.observer) {
                this.observer.observe(counter.element);
            }
        });
    }
}

// Initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
    const counterController = new CounterController();
    window.counterController = counterController;
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { CounterAnimation, CounterController };
}
